//******************************************************************************
//   MSP430x26x Demo - SMBus Master reads from SMBus MSP430 Slave with PEC
//
//   Description: This demo connects two MSP430's via the I2C/SMBus. The master 
//   reads from the slave. This is the master code. The slave code is called
//   msp430x26x_SMB_slave_PEC.c. The data from the slave transmitter begins
//   at 1 and is incremented with each transfer.The UCB0RXIFG interrupt is used 
//   to know when new data has been received.
//   A CRC-8 PEC byte is appended after each data byte by the slave
//   and is validated by the master. If not correct LED on P1.0 is set.
//   ACLK = n/a, MCLK = SMCLK = BRCLK = DCO ~ 1.1MHz
//   //* USCI Required *//
//   
//                                 /|\  /|\
//                  MSP430F2619     10k  10k     MSP430F2619
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.1|<-|----+>|P3.1             |
//             |                 |  |      |             P1.0|-->LED
//             |                 |  |      |                 |
//             |             P3.2|<-+----->|P3.2             |
//             |                 |         |                 |
//
//
//  H. Grewal
//  Texas Instruments Inc.
//  June 2009
//  Built with IAR Embedded Workbench Version: 4.11B
//******************************************************************************

#include  <msp430x26x.h>

#define CRC_MSG_SIZE	4                   // # of Bytes to calculate CRC
#define CRC8_POLY	    0x07                // X^8 + X^2 + X + 1
#define CRC8_INIT_REM   0x0                 // Initial Remainder

unsigned char crc8MakeBitwise(unsigned char crc, unsigned short poly, unsigned char *pmsg, unsigned int msg_size);

unsigned char crc8;                         // CRC-8 result
unsigned char msg[CRC_MSG_SIZE] = {0};      // Array to store Address,Data,CRC
unsigned char RXData = 0;                   // Data Received from Slave            
unsigned char RXByteCtr;                    // # of Bytes Received

void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  P3SEL |= 0x06;                            // Select I2C pins
  P1DIR |= 0xFF;                            // P1.0 output
  P1OUT &= ~0x01;                           // Clear P1.0
                                            // unused ports to output
  P2DIR =  0xFF;
  P3DIR =  0xFF;
  P4DIR =  0xFF;
  P5DIR =  0xFF;
  P6DIR =  0xFF;
  UCB0CTL1 |= UCSWRST;                      // Enable SW reset
  UCB0CTL0 = UCMST + UCMODE_3 + UCSYNC;     // I2C Master, synchronous mode
  UCB0CTL1 = UCSSEL_2 + UCSWRST;            // Use SMCLK, keep SW reset
  UCB0BR0 = 12;                             // fSCL = SMCLK/12 = ~100kHz
  UCB0BR1 = 0;
  UCB0I2CSA = 0x0048;                       // Slave Address is 048h
  UCB0CTL1 &= ~UCSWRST;                     // Clear SW reset, resume operation
  IE2 |= UCB0RXIE;                          // Enable RX interrupt
  while (1)
  {
    RXByteCtr = 2;                          // Load RX byte counter
    while (UCB0CTL1 & UCTXSTP);             // Ensure stop condition got sent
    UCB0CTL1 |= UCTXSTT;                    // I2C start condition
    __bis_SR_register(LPM0_bits + GIE);     // Enter LPM0 w/ interrupts
    crc8 = crc8MakeBitwise(CRC8_INIT_REM, CRC8_POLY, msg, CRC_MSG_SIZE);
    if (crc8 != 0)                          // Incorrect CRC/data received
    P1OUT = 0x01;                           // Turn ON a LED indicator
    else
    P1OUT = 0;
  } 

}
// CRC-8 Algorithm to check the PEC
unsigned char crc8MakeBitwise(unsigned char crc, unsigned short poly, unsigned char *pmsg, unsigned int msg_size)
{
   volatile unsigned char i, j, carry = 0;
   
   for(i = 0 ; i < (msg_size-1) ; i ++)     // For 3 bytes: Address,Data,CRC (from Slave) 
    {
       crc ^= *pmsg++;                      // CRC XOR (Address or Data or CRC)

       for(j = 0 ; j < 8 ; j++)             // Operation on 8-bits
       {
        carry = crc & 0x80;                 // Check if MSB=1
        if (carry)
        {
         crc <<= 1;                         // Left shift CRC to get MSB=1
         crc ^= poly;                       // XOR with CRC POLYNOMIAL 0x07
        }
   
        else
        crc <<= 1;                          // Left shift CRC to get MSB=1
       }
    }
   
    return(crc);                            // Final value returned to the function call 
}

//-------------------------------------------------------------------------------
// The USCI_B0 data ISR is used to move received data from the I2C slave
// to the MSP430 memory. It is structured such that it can be used to receive
// any 2+ number of bytes by pre-loading RXByteCtr with the byte count.
//-------------------------------------------------------------------------------
#pragma vector = USCIAB0TX_VECTOR
__interrupt void USCIAB0TX_ISR(void)
{
  RXByteCtr--;                              // Decrement RX byte counter
  if (RXByteCtr)
  {
    RXData = UCB0RXBUF;                     // RX data
    msg[0] = UCB0I2CSA;                     // CRC to be calculated on Address,R/W
    msg[1] = RXData;                        // and incoming data bytes
    
    if (RXByteCtr == 1)                     // Only one byte left?
    {
      UCB0CTL1 |= UCTXSTP;                  // Generate I2C stop condition
    }
  }
  else
  {
    RXData = UCB0RXBUF;                     // RX data
    msg[2] = RXData;
     
    __bic_SR_register_on_exit(CPUOFF);      // Exit LPM0
  }
     
}

  